#
# Copyright (C) 2006 by Victor Julien <victor@inliniac.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#

#
# Object for communication between a barnyard clone and Sguil.
#


package SguilBarnyardComms;

use fields qw ( sock sid cid name );
use IO::Socket;

sub new {
	my $type = shift;
	my SguilBarnyardComms $self = fields::new(ref $type || $type);
	
	$self->{sock} = undef;
	$self->{sid}  = undef;
	$self->{cid}  = undef;
	$self->{name} = undef;
	
	return $self;
}

#
# handles all tasks related to the building the
# connection.
#
sub connect {
	my SguilBarnyardComms $self = shift;

	my $ip = shift @_;
	my $port = shift @_;
	my $sensor = shift @_;
	
	$self->{name} = $sensor;
	
	# socket connect
	$self->{sock} = new IO::Socket::INET (
			PeerAddr => $ip,
			PeerPort => $port,
			Proto => 'tcp' ) or die $!;
	
	# send our SidCidRequest
	eval { $self->send("SidCidRequest $sensor\n") };
	if($@) { 
		die $@;
	}

	my $resp = "";

	# receive the answer
	eval { $resp = $self->readline() };
	if($@) {
		die $@;
	}

	chomp $resp;
	
	print "resp " . $resp . "\n";
	(my $cmd, $self->{sid}, $self->{cid} ) = split (/ /, $resp);
	if(not $cmd eq "SidCidResponse") {
		die "Expected SidCidResponse, but got: \"$cmd\"";
	}

	#print "sid $self->{sid} cid $self->{cid} name $self->{name}\n";
	return;
}

sub disconnect {
	my SguilBarnyardComms $self = shift;

	if(not defined $self->{sock}) {
		die "Not connected";
	}
	
	close($self->{sock}) or die $!;
	$self->{sock} = undef;
	
	return;
}

#
# Send a real time alert
#
# TODO
# 	use all options if supplied (e.g. tcp options, etc)
# 	require only bare minimum, fall back to sensible defaults
# 	
# 
#
sub rtevent {
	my SguilBarnyardComms $self = shift;

	if(not defined $self->{sock}) {
		die "Not connected";
	}
	
	# get the alert hash from the caller (by reference)
        my $ref = shift @_;
        my %alert = %{$ref};

	# sid, cid and sensor name come from the object
	my $sid = $self->{sid};
	my $cid = $self->{cid};
	my $sensor_name = $self->{name};
	
	#
	# FIRST the required data
	# sip, dip, msg, time
	if(not defined ( $alert{"sipstr"} ) || not defined ( $alert{"dipstr"} ) ) {
		die "missing data: sipstr or dipstr missing.";
	}
	if(not defined ( $alert{"sipdec"} ) || not defined ( $alert{"dipdec"} ) ) {
		die "missing data: sipdec or dipdec missing.";
	}
	if(not defined ( $alert{"msg"} ) ) {
		die "missing data: msg missing.";
	}
	if(not defined ( $alert{"time"} ) ) {
		die "missing data: time missing.";
	}
			
        # IPaddresses
        my $sip_dec = $alert{"sipdec"};
        my $sip_str = $alert{"sipstr"};
        my $dip_dec = $alert{"dipdec"};
        my $dip_str = $alert{"dipstr"};
	# Message
	my $msg = $alert{"msg"};
	# Time
	my $time = $alert{"time"};

	# timestamp
	my $ts = $time;
	if(defined $alert{"ts"} ) {
		$ts = $alert{"ts"};
	}

	# this can only be 0 it seems
	my $status = 0;

	# snort event
	my $snort_event_id = 0;
	if( defined $alert{"snort_event_id"} ) {
		$snort_event_id = $alert{"snort_event_id"};
	}
        my $snort_event_ref = 0;
	if( defined $alert{"snort_event_ref"} ) {
		$snort_event_ref = $alert{"snort_event_ref"};
	}
        my $siggen = 0;
	if( defined $alert{"siggen"} ) {
		$siggen = $alert{"siggen"};
	}
        my $sigid = 0;
	if( defined $alert{"sigid"} ) {
		$sigid = $alert{"sigid"};
	}

	# revision of the rule
	my $rev = 0;
	if( defined $alert{"rev"} ) {
		$rev = $alert{"rev"};
	}
	# priority
	my $prio = 0; # DEBUG, lowest
	if ( defined $alert{"prio"} ) {
		$prio = $alert{"prio"};
	}
	# class
        my $class = "unknown"; # defined in snort manual
	if ( defined $alert{"class"} ) {
		$class = $alert{"class"};
	}


	# IP header
	my $ipver = 4; # default to IPv4.
	if(defined($alert{"ipver"})) {
        	$ipver = $alert{"ipver"};
	}
        my $ipproto = 6; # default to TCP.
	if(defined($alert{"ipproto"})) {
		$ipproto = $alert{"ipproto"};
	}
	my $ip_hlen = 0;
	if(defined($alert{"ip_hlen"})) {
		$ip_hlen = $alert{"ip_hlen"};
	}
	my $ip_tos = 0;
	if(defined($alert{"ip_tos"})) {
		$ip_tos = $alert{"ip_tos"};
	}
	my $ip_len = 0;
	if(defined($alert{"ip_len"})) {
		$ip_len = $alert{"ip_len"};
	}
	my $ip_id = 0;
	if(defined($alert{"ip_id"})) {
		$ip_id = $alert{"ip_id"};
	}
	my $ip_flags = 0;
	if(defined($alert{"ip_flags"})) {
		$ip_flags = $alert{"ip_flags"};
	}
	my $ip_off = 0;
	if(defined($alert{"ip_off"})) {
		$ip_off = $alert{"ip_off"};
	}
	my $ip_ttl = 0;
	if(defined($alert{"ip_ttl"})) {
		$ip_ttl = $alert{"ip_ttl"};
	}
	my $ip_csum = 0;
	if(defined($alert{"ip_csum"})) {
		$ip_csum = $alert{"ip_csum"};
	}
	
        # ICMP parameters
	my $icmp_type = 0;
	if(defined($alert{"icmp_type"})) {
		$icmp_type = $alert{"icmp_type"};
	}
	my $icmp_code = 0;
	if(defined($alert{"icmp_code"})) {
		$icmp_code = $alert{"icmp_code"};
	}
	my $icmp_csum = 0;
	if(defined($alert{"icmp_csum"})) {
		$icmp_csum = $alert{"icmp_csum"};
	}
	my $icmp_id = 0;
	if(defined($alert{"icmp_id"})) {
		$icmp_id = $alert{"icmp_id"};
	}
	my $icmp_seq = 0;
	if(defined($alert{"icmp_seq"})) {
		$icmp_seq = $alert{"icmp_seq"};
	}

	# TCP/UDP ports
        my $sp = 0;
	if(defined $alert{"sp"}) {
		$sp = $alert{"sp"};
	}
        my $dp = $alert{"dp"};
	if(defined $alert{"dp"}) {
		$dp = $alert{"dp"};
	}

	# TCP header
	my $tcp_seq = 0;
	if(defined $alert{"tcp_seq"}) {
		$tcp_seq = $alert{"tcp_seq"};
	}
	my $tcp_ack = 0;
	if(defined $alert{"tcp_ack"}) {
		$tcp_ack = $alert{"tcp_ack"};
	}
	my $tcp_off = 0;
	if(defined $alert{"tcp_off"}) {
		$tcp_off = $alert{"tcp_off"};
	}
	my $tcp_res = 0;
	if(defined $alert{"tcp_res"}) {
		$tcp_res = $alert{"tcp_res"};
	}
	my $tcp_flags = 0;
	if(defined $alert{"tcp_flags"}) {
		$tcp_flags = $alert{"tcp_flags"};
	}
	my $tcp_win = 0;
	if(defined $alert{"tcp_win"}) {
		$tcp_win = $alert{"tcp_win"};
	}
	my $tcp_csum = 0;
	if(defined $alert{"tcp_csum"}) {
		$tcp_csum = $alert{"tcp_csum"};
	}
	my $tcp_urp = 0;
	if(defined $alert{"tcp_urp"}) {
		$tcp_urp = $alert{"tcp_urp"};
	}

	# UDP header
	my $udp_len = 0;
	if(defined $alert{"udp_len"}) {
		$udp_len = $alert{"udp_len"};
	}
	my $udp_csum = 0;
	if(defined $alert{"udp_csum"}) {
		$udp_csum = $alert{"udp_csum"};
	}

        # the attack payload
        my $payload = "";
	if(defined $alert{"payload"}) {
		$payload = $alert{"payload"};
	}

        #print $modsec_payload . "\n";

        # assemble the string to send to the sensor.
        #
        #           0       1       2    3    4            5              
	my $str = "RTEVENT $status $sid $cid $sensor_name " . 
	# 5               6                  7       8       9
	"$snort_event_id $snort_event_ref \{$time\} $siggen $sigid " .
	# 10    11       12    13    14  
	"$rev \{$msg\} \{$ts\} $prio $class " .
        # 15       16       17       18
        "$sip_dec $sip_str $dip_dec $dip_str " .
        # 19       20     21       22      23      24     25        26      27      28
        "$ipproto $ipver $ip_hlen $ip_tos $ip_len $ip_id $ip_flags $ip_off $ip_ttl $ip_csum " .
	# 29         30         31         32       33
	"$icmp_type $icmp_code $icmp_csum $icmp_id $icmp_seq " . 
	# 34  35
	"$sp $dp " .
        # 36       37       38       39       40         41       42        43
        "$tcp_seq $tcp_ack $tcp_off $tcp_res $tcp_flags $tcp_win $tcp_csum $tcp_urp " .
	# 44       45
	"$udp_len $udp_csum " .
	#   46
	"\{$payload\}\n";

	# Send the alert to the agent
	eval { $self->send($str) };
	if($@) {
		die $@;
	}

	# Read the response line
	my $resp = "";
	eval { $resp = $self->readline() };
	if($@) {
		die $@;
	}
	chomp($resp);
	
	# evaluate it. We expect a line like:
	# Confirm 1854
	# where 1854 is the alert cid.
	if($resp =~ /^Confirm \d+/)
	{
		my @parsed = ($resp =~ /^Confirm (\d+)$/);
		if(@parsed == 1) {
			(my $resp_cid) = @parsed;

			if($cid != $resp_cid) {
				die "Response cid $resp_cid does not match alert cid $cid";
			}
		} else {
			die "Parsing agent response failed for: $resp";
		}
	} else {
		die "Agent returned error: $resp";
	}

	return;
}

sub readline {
	my SguilBarnyardComms $self = shift;

	if(not defined $self->{sock}) {
		die "Not connected";
	}

	return readline ( $self->{sock} );
}

sub send {
	my SguilBarnyardComms $self = shift;
	my $prt = shift;

	if(not defined $self->{sock}) {
		die "Not connected";
	}

	$self->{sock}->send("$prt") or die "Send failed: $!\n";
}

sub getcid {
	my SguilBarnyardComms $self = shift;

	return $self->{cid};
}

sub incrcid {
	my SguilBarnyardComms $self = shift;

	return $self->{cid}++;
}

sub getsid {
	my SguilBarnyardComms $self = shift;

	return $self->{sid};
}

1;

